/**
* \file: Server.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: CarPlay
*
* \author: J. Harder / ADIT/SW1 / jharder@de.adit-jv.com
*
* \copyright (c) 2013-2014 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#include <unistd.h>
#include <systemd/sd-daemon.h>
#include "Common.h"
#include "Server.h"
#include "Session.h"
#include "utils/Utils.h"
#include "control/PfCfgConfiguration.h"
#include "utils/Statistics.h"

#include <fstream>

#include "AirPlayVersion.h"

extern CarPlayControlClientRef                 gCarPlayControlClient;
using namespace std;

namespace adit { namespace carplay
{

const uint64_t carPlayFeatures = 0x104000280;

enum DisplayInputFeature
{
    DisplayInputFeature_Knobs                     = 0x01,
    DisplayInputFeature_LowFidelityTouch          = 0x02,
    DisplayInputFeature_HighFidelityTouch         = 0x04,
    DisplayInputFeature_Knobs_LowFidelityTouch    = 0x08,
    DisplayInputFeature_Knobs_HighFidelityTouch   = 0x10
};

static OSStatus getIconSize(const char* iconPath, int* width, int* height);

Server::Server()
{
    shutdownSignal = false;
    sessionCreated = false;
    config = nullptr;
    mainScreen = 0;
}

Server::~Server()
{
    dipo_safe_delete(config);
}

bool Server::Initialize()
{
    LOGD_DEBUG((dipo, "CarPlay git tag                     : %s", CarPlaySourceVersion.c_str()));
    LOGD_DEBUG((dipo, "Communications plug-in core version : %s",
            CarPlayCommunicationsPluginVersion.c_str()));
    LOGD_DEBUG((dipo, "developed with iOS                  : %s",
            CarPlayDevelopedWithiOSVersion.c_str()));
    auto pfcfg = PfCfgConfiguration::FromFile("/etc/pfcfg/carplay.cfg");
    config = pfcfg;
    if (config == nullptr)
    {
        LOG_ERROR((dipo, "could not open /etc/pfcfg/carplay.cfg"));
#if 0 // TODO until name change has settled down keep the fallback
        return false;
#else // try dipo.cfg
        config = PfCfgConfiguration::FromFile("/etc/pfcfg/dipo.cfg");
        if (config == nullptr)
            return false;
#endif
    }
    else
    {
        // for debugging purposes
        Statistics::Instance().AddServer(string("/etc/pfcfg/carplay.cfg"), *pfcfg);
    }

    if (!pluginManager.Initialize(*config))
    {
        // errors logged
        return false;
    }

    if (!initCarPlayDeviceAdapter())
    {
        LOG_ERROR((dipo, "failed to initialize CarPlay Device Adapter"));
        return false;
    }

    // register control delegates
    AirPlayReceiverServerDelegate delegate;
    AirPlayReceiverServerDelegateInit(&delegate);
    delegate.control_f          = HandleControl;
    delegate.copyProperty_f     = HandleCopyProperty;
    delegate.setProperty_f      = HandleSetProperty;
    delegate.sessionCreated_f   = HandleSessionCreated;
    delegate.sessionFailed_f    = HandleSessionFailed;
    AirPlayReceiverServerSetDelegate(reference, &delegate);

    // register screen
    OSStatus err = ScreenCreate(&mainScreen, nullptr);
    if (err == kNoErr)
    {
        if( !RegisterScreenSettings())
            return false; //error logged
    }
    else
    {
        LOG_ERROR((dipo, "ScreenCreate failed"));
        return false;
    }
    LOGD_DEBUG((dipo, "CarPlay Server Initialization successfully"));
    return true;
}

int Server::Main(int argc, const char** argv)
{
    (void)argc;
    (void)argv;

    LOGD_DEBUG((dipo, "in Server::Main"));
    while (!shutdownSignal)
    {
        // TODO proper exit handling
        sleep(1);
    }

    LOGD_DEBUG((dipo, "exiting Server::Main"));
    return true;
}

void Server::SignalShutdown()
{
    LOGD_DEBUG((dipo, "Server::SignalShutdown"));
    shutdownSignal = true;
}

void Server::RemoveSession(Session& inSession)
{
    auto ref = inSession.GetReference();

    // remove session if not already removed by failed init process
    if (Session::Has(ref))
    {
        LOGD_DEBUG((dipo, "Remove session with IControlAdapter"));
        Session::Remove(ref);
    }
    sessionCreated = false;
    sessions.remove(&inSession);
}

OSStatus Server::HandleControl(AirPlayReceiverServerRef inServer, CFStringRef inCommand,
        CFTypeRef inQualifier, CFDictionaryRef inParams, CFDictionaryRef* outParams,
        void* inContext)
{
    (void)inServer;
    (void)inQualifier;
    (void)inParams;
    (void)outParams;
    (void)inContext;

    auto server = Get(inServer);
    if (server == nullptr)
    {
        LOG_WARN((dipo, "Server already removed %p", inServer));
        return kUnknownErr;
    }

    char str[1024];
    CFGetCString(inCommand, str, 1024);


    if (false)
    {
        //
    }
    else if (CFEqual(inCommand, CFSTR(kAirPlayCommand_StartServer)))
    {
        sd_notify(0, "READY=1\nSTATUS=Waiting for session...");
        LOG_INFO((dipo, "CarPlay server running"));
    }
    else if (CFEqual(inCommand, CFSTR(kAirPlayCommand_StopServer)))
    {
#if 0 // server/session cleanup will be done in _AirPlayReceiverServerPlatformFinalize and HandleSessionFinalized
        std::list<Session *>::const_iterator session_iterator;
        auto list_session =  server->sessions;

        if(!list_session.empty())
            LOG_WARN((dipo, "CarPlay server is being stopped, but %d sessions are still running", list_session.size()));

        for (session_iterator = list_session.begin(); session_iterator != list_session.end(); ++session_iterator)
        	server->RemoveSession(**session_iterator);

        Server::Remove(inServer);
#endif
        sd_notify(0, "STATUS=Server stopped...");
        LOG_INFO((dipo, "CarPlay server stopped"));
    }
    else if( CFEqual( inCommand, CFSTR( kAirPlayCommand_DisableBluetooth ) ) )
    {
        LOG_WARN((dipo, "HandleControl Disable Bluetooth session control request\n"));
    }
    else
    {
    	LOG_WARN((dipo, "Server::HandleControl: %s not handled", str));
    	return kNotHandledErr;
    }

    return kNoErr;
}

CFTypeRef Server::HandleCopyProperty(AirPlayReceiverServerRef inServer, CFStringRef inProperty,
        CFTypeRef inQualifier, OSStatus* outErr, void* inContext)
{
    (void)inQualifier;
    (void)inContext;

    auto me = Get(inServer);
    auto config = me->config;

    OSStatus status = kNoErr;
    CFTypeRef value = nullptr;

    char str[1024];
    CFGetCString(inProperty, str, 1024);

    if (false)
    {
    }
    else if (CFEqual(inProperty, CFSTR(kAirPlayKey_OSInfo)))
    {
        auto OSInfo = config->GetItem("core-osInfo", "");
        if ((value = (CFTypeRef) CFStringCreateFromStdString(OSInfo)) == nullptr)
        {
            status = kNoMemoryErr;
        }
    }
    else if (CFEqual(inProperty, CFSTR(kAirPlayProperty_BluetoothIDs)))
    {
    	if (!me->sessions.empty())
    	{
            auto anySession = me->sessions.front();
            if (anySession != nullptr)
            {
                list < string > listBluetoothIDs = anySession->GetBluetoothIDs();
                if ((value = (CFTypeRef) CFArrayCreateMutableFromStdList(listBluetoothIDs)) == nullptr)
                {
                    status = kNoMemoryErr;
                }

                for (auto item : listBluetoothIDs)
                    LOGD_DEBUG((dipo, "INFO Bluetooth ID: %s", item.c_str()));
            }
            else
            {
                LOG_ERROR((dipo, "no active Session to retrieve Bluetooth IDs"));
            }
    	}
    	else
    	{
    		LOG_ERROR((dipo, "There is no active Session Available"));
    	}
    }
    else if (CFEqual(inProperty, CFSTR(kAirPlayKey_HIDLanguages)))
    {
        auto hidLanguages = config->GetItems("hid-language-supported");
        if(!hidLanguages.empty())
        {
            if ((value = (CFTypeRef) CFArrayCreateMutableFromStdList(hidLanguages)) == nullptr)
            {
                status = kNoMemoryErr;
            }
        }
    }
    else if (CFEqual(inProperty, CFSTR(kAirPlayProperty_PlaybackAllowed)))
    {
        // presumably if it is allowed to use AirPlay
        // TODO exact meaning of this property is unclear
        uint64_t number = config->GetNumber("core-playbackAllowed", 1);
        value = (CFTypeRef) CFNumberCreateInt64(number);
    }
    else if (CFEqual(inProperty, CFSTR(kAirPlayKey_NightMode)))
    {
    	if (!me->sessions.empty())
    	{
            auto anySession = me->sessions.front();
            if (anySession != nullptr)
            {
                NightMode nightMode = anySession->GetNightMode();
                if (nightMode != NightMode_NotSupported)
                {
                    value = (CFTypeRef)CFBooleanCreateFromInt(nightMode);
                }
                else
                {
                    LOGD_DEBUG((dipo, "NightMode property not set/supported"));
                    status = kUnsupportedErr;
                }
            }
            else
            {
                LOG_ERROR((dipo, "no active Session to retrieve night mode"));
            }
    	}
    	else
    	{
    	    LOG_ERROR((dipo, "There is no active Session Available"));
    	}
    }
    else if (CFEqual(inProperty, CFSTR(kAirPlayKey_Features)))
    {
        if ((value = (CFTypeRef) CFNumberCreateFromStdInt(carPlayFeatures)) == nullptr)
        {
            status = kNoMemoryErr;
        }
    }
    else if (CFEqual(inProperty, CFSTR(kAirPlayProperty_OEMIconVisible)))
    {
        auto oemIconVisible = config->GetItem("core-oemIconVisible", "0");
        value = (CFTypeRef) CFStringCreateFromStdString(oemIconVisible);
    }
    else if (CFEqual(inProperty, CFSTR(kAirPlayKey_RightHandDrive)))
    {
        auto rightHandDrive = config->GetItem("core-rightHandDrive", "0");
        value = (CFTypeRef) CFStringCreateFromStdString(rightHandDrive);
    }
    else if (CFEqual(inProperty, CFSTR(kAirPlayKey_LimitedUI)))
    {
        // LimitedUI will be supported only if limited-ui-elements present
        auto uiElements = config->GetItems("limited-ui-elements");
        if(!uiElements.empty())
        {
            if (!me->sessions.empty())
            {
                auto anySession = me->sessions.front();
                if (anySession != nullptr)
                {
                    value = (CFTypeRef)CFBooleanCreateFromInt(anySession->GetLimitedUI());
                    anySession->SetSupportLimitedUI(true);
                }
                else
                {
                    LOG_ERROR((dipo, "no active Session to retrieve limited UI mode"));
                }
            }
            else
            {
                LOG_ERROR((dipo, "There is no active Session Available"));
            }
        }
        else
        {
            LOGD_DEBUG((dipo, "LimitedUI not supported"));
        }
    }
    else if (CFEqual(inProperty, CFSTR(kAirPlayKey_LimitedUIElements)))
    {
        /*
         * if there is no limited-ui-elements
         * then empty CFMutableArrayRef is returned
         * from CFArrayCreateMutableFromStdList
         */
        auto uiElements = config->GetItems("limited-ui-elements");
        if(!uiElements.empty())
        {
            if ((value = (CFTypeRef) CFArrayCreateMutableFromStdList(uiElements)) == nullptr)
            {
                status = kNoMemoryErr;
            }
        }
    }
    else if (CFEqual(inProperty, CFSTR(kAirPlayProperty_OEMIcon)))
    {
        // already handled in configuration, ignore
    }
    else if (CFEqual(inProperty, CFSTR(kAirPlayProperty_OEMIconLabel)))
    {
        // already handled in configuration, ignore
    }
    else if(CFEqual(inProperty, CFSTR(kAirPlayProperty_OEMIcons)))
    {
        auto oemIcons = config->GetItems("oemIcons");
        if(!oemIcons.empty())
        {
            CFMutableArrayRef IconsArray = CFArrayCreateMutable(nullptr, 0, &kCFTypeArrayCallBacks);
            dipo_exit_on_null(IconsArray);

            for (auto line : oemIcons)
            {
                std::map<string, string> oemIconProperties = ParseMap(line, true, "oemIcons");
                // Set ImageData from given file path
                std::map<string,string>::iterator iter = oemIconProperties.find("path");
                if(iter != oemIconProperties.end())
                {
                    CFMutableDictionaryRef    dict;

                    CFDataRef IconPath = CFDataCreateWithFilePath(iter->second.c_str(), NULL);
                    if(IconPath != nullptr)
                    {
                        dict = CFDictionaryCreateMutable( NULL, 0, &kCFTypeDictionaryKeyCallBacks, &kCFTypeDictionaryValueCallBacks );
                        if(dict == nullptr) *outErr = kNoMemoryErr;

                        CFArrayAppendValue( IconsArray, dict );

                        CFDictionarySetValue( dict, CFSTR( kAirPlayOEMIconKey_ImageData ), IconPath );
                        CFRelease( IconPath );

                        // read Image height and width in pixes from given file path and set
                        int iconWidth=0, iconHeight=0;
                        OSStatus err = getIconSize(iter->second.c_str(), &iconWidth, &iconHeight);
                        LOG_INFO((dipo, "Set OemIcon: %s resolution: %dx%d",iter->second.c_str(), iconWidth,iconHeight));
                        if(err == kNoErr) {
                            CFDictionarySetInt64( dict, CFSTR( kAirPlayOEMIconKey_HeightPixels ), iconWidth);
                            CFDictionarySetInt64( dict, CFSTR( kAirPlayOEMIconKey_WidthPixels ), iconHeight);
                        }
                    }
                    else
                    {
                        LOG_WARN((dipo, "Please check the oemIcons(Path) defined in cfg, %s not valid path", iter->second.c_str()));
                        continue;
                    }
                    // Set prerendered key-value
                    iter = oemIconProperties.find("full-bleed");
                    if(iter != oemIconProperties.end())
                    {
                        CFDictionarySetValue( dict, CFSTR( kAirPlayOEMIconKey_Prerendered ), (strcmp(iter->second.c_str(), "true") == 0) ? kCFBooleanTrue : kCFBooleanFalse);
                    }
                    else
                    {
                        LOG_WARN((dipo, "prerendered key(full-bleed) is not set in oemIcons-*"));
                    }
                    CFRelease(dict);
                }
            }

            value = IconsArray;
        }
        else
        {
            if(0 != config->GetNumber("core-oemIconVisible", 0))
                LOG_WARN((dipo, "oemIconVisible set to 1 but no oemIcons-* defined in cfg"));

                *outErr = kNotFoundErr;
                value = nullptr;
        }
    }
    else if(CFEqual(inProperty,CFSTR(kAirPlayProperty_ExtendedFeatures)))
    {
        auto extendedFeatures = config->GetItems("extended-features");
        if (!extendedFeatures.empty())
        {
            if ((value = (CFTypeRef) CFArrayCreateMutableFromStdList(extendedFeatures)) == nullptr)
            {
                status = kNoMemoryErr;
            }
        }
    }
    else if(CFEqual(inProperty, CFSTR(kAirPlayKey_VehicleInformation)))
    {
        if (!me->sessions.empty())
        {
            auto anySession = me->sessions.front();
            if (anySession != nullptr)
            {

                bool updateVehicleInfo = anySession->GetUpdateVehicleInfo();
                if (updateVehicleInfo == true)
                {
                    ETCInfo etcSupport = anySession->GetVehicleETCSupport();
                    NADInfo nadSupport = anySession->GetVehicleNADSupport();

                    auto vehicleInfo = CFDictionaryCreateMutable(NULL, 0, &kCFTypeDictionaryKeyCallBacks,
                                    &kCFTypeDictionaryValueCallBacks);

                    if(etcSupport != ETCInfo_NotSupported)
                    {
                        auto params = CFDictionaryCreateMutable(NULL, 0, &kCFTypeDictionaryKeyCallBacks,
                                   &kCFTypeDictionaryValueCallBacks);

                        CFDictionarySetValue( params, CFSTR( kAirPlayKey_Active ), etcSupport ? kCFBooleanTrue : kCFBooleanFalse );

                        CFDictionarySetValue( vehicleInfo, CFSTR( kAirPlayVehicleInformation_ETC ), params );
                        ForgetCF( &params );
                    }

                    if(nadSupport != NADInfo_NotSupported)
                    {
                        auto params = CFDictionaryCreateMutable(NULL, 0, &kCFTypeDictionaryKeyCallBacks,
                                                       &kCFTypeDictionaryValueCallBacks);

                        CFDictionarySetBoolean( params, CFSTR( kAirPlayKey_Active ), nadSupport ? true : false );

                        CFDictionarySetValue( vehicleInfo, CFSTR( kAirPlayVehicleInformation_NavigationAidedDriving ), params );

                        ForgetCF( &params );
                    }

                value = (CFTypeRef)vehicleInfo;
                }
            }
            else
            {
                LOG_ERROR((dipo, "no active Session to update VehicleInformation in info response"));
            }
        }
        else
        {
            LOG_ERROR((dipo, "There is no active Session Available"));
        }
    }

    else
    {
        LOG_WARN((dipo, "Server::CopyProperty: %s not handled", str));
        status = kNotHandledErr;
    }

    if (outErr)
    {
        *outErr = status;
    }

    return value;
}

OSStatus Server::HandleSetProperty(AirPlayReceiverServerRef inServer, CFStringRef inProperty,
        CFTypeRef inQualifier, CFTypeRef inValue, void* inContext)
{
    (void)inServer;
    (void)inProperty;
    (void)inQualifier;
    (void)inValue;
    (void)inContext;

    char str[1024];
    CFGetCString(inProperty, str, 1024);

    LOG_WARN((dipo, "Server::HandleSetProperty: %s not handled", str));
    return kNotHandledErr;
}

void Server::HandleSessionCreated(AirPlayReceiverServerRef inServer,
        AirPlayReceiverSessionRef inSession, void* inContext)
{
    (void)inContext;

    dipo_return_on_invalid_argument(dipo, inServer == nullptr);
    auto me = Get(inServer);
    LOGD_DEBUG((dipo, "Server::HandleSessionCreated ref=%p", inSession));

    if (me != nullptr)
    {
#if 0 // TODO workaround
        while (!me->sessions.empty())
            Session::Remove(me->sessions.front()->GetReference());

        // dirty: give layermanager a chance to clean-up
        sleep(2);
#endif

        auto session = Session::Add(inSession);
        if (session == nullptr)
        {
            LOG_ERROR((dipo, "failed to create a new session"));
        }
        else
        {
            if (!session->Start(*me, *me->config))
            {
                LOG_ERROR((dipo, "failed to start a new session"));

                Session::Remove(inSession); // ignore error
            }
            else
            {
                me->sessions.push_back(session);
                me->sessionCreated = true;
            }
        }

        //Re-Register screen settings if changed
        if(me->IsScreenSettingsChanged())
        {
            LOGD_DEBUG((dipo, "Screen settings changed. Re-register screen settings"));

            // Deregister screen settings
            ScreenDeregister(me->mainScreen);

            /* If Screen Register fails during session startup, default values 960 x 540 will be
               sent to Apple device */
            me->RegisterScreenSettings(); //error logged
        }
        else
        {
            LOGD_DEBUG((dipo, "Screen settings not changed"));
        }
    }
}

void Server::HandleSessionFailed(AirPlayReceiverServerRef inServer, OSStatus inReason, void * inContext)
{
    (void)inContext;
    (void)inServer;

    LOGD_DEBUG((dipo, "Server::HandleSessionFailed, error %d", inReason));

    // do not perform the cleanup here, HandleControl with parameter kAirPlayCommand_StopSession will be called after
    // this function returned to do the cleanup.
}

CAEStatus Server::StartCarPlayDeviceDiscovery(void)
{
    OSStatus err;
    CAEStatus adapterError = CarPlayAdapterError_NoError;
    err = CarPlayControlClientStart( gCarPlayControlClient );
    if( err != kNoErr )
    {
        adapterError = CarPlayAdapterError_Generic;
    }
    LOG_INFO((dipo, "StartCarPlayDeviceDiscovery is Called"));
    return(adapterError);

}

CAEStatus Server::StopCarPlayDeviceDiscovery()
{
    OSStatus err = kNoErr;
    CAEStatus adapterError = CarPlayAdapterError_NoError;
    CarPlayControlClientStop( gCarPlayControlClient );
    if( err != kNoErr )
    {
        adapterError = CarPlayAdapterError_Generic;
    }
    LOG_INFO((dipo, "StopCarPlayDeviceDiscovery is Called"));
    return(adapterError);

}

CAEStatus Server::ConnectDevice(const char *deviceId)
{
    OSStatus err;
    const char *cStrDeviceName = NULL;
    CFStringRef name = NULL;
    char *storage = NULL;
    CAEStatus adapterError = CarPlayAdapterError_NoError;
    if(deviceId == nullptr)
    {
        LOG_ERROR((dipo, "Server::ConnectDevice not valid deviceId"));
        return(CarPlayAdapterError_ParamErr);
    }

    if(!this->sessionCreated)
    {
        CarPlayControllerRef outDevice = NULL;
        char *cStr = NULL;
        outDevice = SearchDevice(deviceId, &cStr);

        if(cStr != NULL && outDevice != NULL)
        {
            CarPlayControllerCopyName(outDevice, &name);
            CFStringGetOrCopyCStringUTF8(name, &cStrDeviceName, &storage, NULL);
            LOG_INFO((dipo, "Connecting The CarPlay Session On %s Device",cStrDeviceName));

            err = CarPlayControlClientConnect(gCarPlayControlClient, outDevice);// Plugin core API has called
            if( err != kNoErr )
            {
                adapterError = CarPlayAdapterError_Generic;
            }
            free ((char*)cStr);
        }
        else
        {
            adapterError = CarPlayAdapterError_ParamErr;
        }
    }
    else
    {
        adapterError = CarPlayAdapterError_ActiveSession;
        LOG_ERROR((dipo, "Starting New Session While Acive CarPlay Session Exist Is Not Allowed"));
    }

    return(adapterError);
}

CAEStatus Server::DeviceLeftFromAP(const char *deviceId, const char* ipAddress)
{
    OSStatus err;
    CAEStatus adapterError = CarPlayAdapterError_NoError;
    if((deviceId == nullptr) || (ipAddress == nullptr))
    {
        LOG_ERROR((dipo, "Server::DeviceLeft not valid deviceId or ipaddress"));
        return CarPlayAdapterError_ParamErr;
    }

    CarPlayControllerRef outDevice = NULL;
    char *cStr = NULL;
    outDevice = SearchDevice(deviceId, &cStr);

    if((outDevice != NULL) && (cStr != NULL))
    {
        LOGD_DEBUG((dipo, "carplay device %s with ip %s left from Access Point", cStr, ipAddress));
        err = CarPlayControlClientSTALeft(gCarPlayControlClient, outDevice, ipAddress);
        if( err != kNoErr )
        {
            adapterError = CarPlayAdapterError_Generic;
        }
        free ((char*)cStr);
    }
    else
    {
        //already logged(no device found with the provided Mac address)
        adapterError = CarPlayAdapterError_Generic;
    }

    return adapterError;
}

CAEStatus Server::DisconnectDevice(const char *deviceId)
{
    OSStatus err;
    CAEStatus adapterError = CarPlayAdapterError_NoError;
    CarPlayControllerRef outDevice = NULL;
    char *cStr = NULL;
    if(deviceId == nullptr)
    {
        LOG_ERROR((dipo, "Server::DisconnectDevice not valid deviceId"));
        return(CarPlayAdapterError_ParamErr);
    }
    outDevice = SearchDevice(deviceId, &cStr);
    if(cStr != NULL && outDevice != NULL)
    {
        LOG_INFO((dipo, "Disconnecting the CarPlay Session on %s Device",cStr));

        err = CarPlayControlClientDisconnect(gCarPlayControlClient, outDevice);// Plugin core API has called
        if( err != kNoErr )
        {
            adapterError = CarPlayAdapterError_Generic;
        }
        free ((char*)cStr);
    }
    else
    {
        adapterError = CarPlayAdapterError_ParamErr;
    }
    return(adapterError);

}

CarPlayControllerRef Server::SearchDevice(const char *deviceId, char **outStr)
{

    auto it = carplayDeviceMap.find(deviceId);

    if(it == carplayDeviceMap.end())
    {
        LOG_ERROR((dipo, "Server::SearchDevice Couldn't Find The Device"));
        return(NULL);

    }
    char *storage = NULL;
    const char *cStr = NULL;
    CFStringRef name = NULL;
    CarPlayControllerCopyName(it->second, &name);
    CFStringGetOrCopyCStringUTF8(name, &cStr, &storage, NULL);
    *outStr = (char*)calloc(strlen(cStr)+1,sizeof(char));
    strncpy(*outStr, cStr, strlen(cStr));

    LOGD_DEBUG((dipo, "Searching for %s Device",*outStr));

    CFRelease(name);
    free( storage );
    return(it->second);
}

bool Server::UpdateDevicemap(std::string deviceId, CarPlayControllerRef inController, CarPlayDevice *deviceCarplay, bool inDeviceDiscovered)
{
    bool outStatus = true;
    auto it = carplayDeviceMap.find(deviceId);

    if(inDeviceDiscovered)
    {
        if(it != carplayDeviceMap.end())
        {
            LOG_INFO((dipo, "Device is already exist in the map"));
            outStatus = false;
        }
        else
        {
        carplayDeviceMap.insert(std::pair<std::string,CarPlayControllerRef>(deviceId, inController));
        LOG_INFO((dipo, "UpdateDevicemap Device Added"));
        }
    }
    else
    {
        if(it == carplayDeviceMap.end())
        {
            LOG_ERROR((dipo, "Device is not exist in the map\n"));
            outStatus = false;
        }
        else
        {
        carplayDeviceMap.erase(it);
        LOG_INFO((dipo, "UpdateDevicemap Device deleted"));
        }

    }
    carplayDeviceAdapter->OnDeviceUpdated(*deviceCarplay,inDeviceDiscovered);
    free(deviceCarplay->deviceName);
    free(deviceCarplay->btMac);
    free(deviceCarplay->deviceVersion);
    delete(deviceCarplay);
    return outStatus;
}

// ====== private methods ======

bool Server::initCarPlayDeviceAdapter()
{
    carplayDeviceAdapter = move(createAdapter<ICarPlayDeviceAdapter>("ICarPlayDeviceAdapter"));
    if (carplayDeviceAdapter != nullptr)
    {
        if(!carplayDeviceAdapter->Initialize(*this))
        {
            LOG_ERROR((dipo, "ICarPlayDeviceAdapter initialization failed"));
            carplayDeviceAdapter.reset(nullptr);
            return false;
        }
    }
    else
    {
        LOG_ERROR((dipo, "ICarPlayDeviceAdapter not initialized"));
        return false;
    }

    return true;
}

template<class T> unique_ptr<T> Server::createAdapter(const std::string& inName)
{
    string impl = config->GetItem(inName, "");
    unique_ptr<T> ptr(static_cast<T*>(Factory::Instance()->Create(impl)));

    if (ptr == nullptr)
    {
        LOG_ERROR((dipo, "%s implementation %s not found! inside Server", inName.c_str(), impl.c_str()));
        return nullptr;
    }

    return move(ptr);
}

bool Server::IsScreenSettingsChanged()
{
    bool retErr = false;

    if((widthPixels        != config->GetNumber("display-width", 800)) ||
       (heightPixels       != config->GetNumber("display-height", 480)) ||
       (widthMm            != config->GetNumber("display-width-millimeter", 800)) ||
       (heightMm           != config->GetNumber("display-height-millimeter", 480)) ||
       (uuid               != config->GetItem("display-uuid", "")) ||
       (maxFps             != config->GetNumber("display-fps", 30)) ||
       (input              != config->GetNumber("display-input", DisplayInputFeature_HighFidelityTouch) ||
       (primaryInputDevice != config->GetNumber("display-primary-input", kScreenPrimaryInputDevice_TouchScreen))))
    {
        retErr = true;
    }

    return retErr;
}

bool Server::RegisterScreenSettings()
{
    bool retErr;
    OSStatus err;
    input = config->GetNumber("display-input", DisplayInputFeature_HighFidelityTouch);

    uint32_t screenFeatures = 0;
    if ((input & DisplayInputFeature_Knobs) != 0)
        screenFeatures = kScreenFeature_Knobs;
    if ((input & DisplayInputFeature_LowFidelityTouch) != 0)
        screenFeatures = kScreenFeature_LowFidelityTouch;
    if ((input & DisplayInputFeature_HighFidelityTouch) != 0)
        screenFeatures = kScreenFeature_HighFidelityTouch;
    if ((input & DisplayInputFeature_Knobs_LowFidelityTouch) != 0)
        screenFeatures = (kScreenFeature_Knobs | kScreenFeature_LowFidelityTouch);
    if ((input & DisplayInputFeature_Knobs_HighFidelityTouch) != 0)
        screenFeatures = (kScreenFeature_Knobs | kScreenFeature_HighFidelityTouch);

    widthPixels        = config->GetNumber("display-width", 800);
    heightPixels       = config->GetNumber("display-height", 480);
    widthMm            = config->GetNumber("display-width-millimeter", 800);
    heightMm           = config->GetNumber("display-height-millimeter", 480);
    uuid               = config->GetItem("display-uuid", "");
    maxFps             = config->GetNumber("display-fps", 30);
    primaryInputDevice = config->GetNumber("display-primary-input", kScreenPrimaryInputDevice_TouchScreen);

    ScreenSetPropertyInt64(mainScreen, kScreenProperty_Features, NULL, screenFeatures);
    ScreenSetPropertyInt64(mainScreen, kScreenProperty_MaxFPS, NULL, maxFps);
    ScreenSetPropertyInt64(mainScreen, kScreenProperty_WidthPixels, NULL, widthPixels);
    ScreenSetPropertyInt64(mainScreen, kScreenProperty_HeightPixels, NULL, heightPixels);
    ScreenSetPropertyInt64(mainScreen, kScreenProperty_WidthPhysical, NULL, widthMm);
    ScreenSetPropertyInt64(mainScreen, kScreenProperty_HeightPhysical, NULL, heightMm);
    ScreenSetPropertyCString(mainScreen, kScreenProperty_UUID, NULL, uuid.c_str(),
            kSizeCString);
    ScreenSetPropertyInt64(mainScreen, kScreenProperty_PrimaryInputDevice, NULL, primaryInputDevice);

    err = ScreenRegister(mainScreen);
    if (err != kNoErr)
    {
        LOG_ERROR((dipo, "ScreenRegister failed"));
        retErr = false;
    }
    else
    {
        LOGD_DEBUG((dipo, "ScreenRegister success"));
        retErr = true;
    }

    return retErr;
}

static OSStatus getIconSize(const char* iconPath, int* width, int* height)
{
    OSStatus err = kNoErr;
    ifstream file;
    file.open(iconPath);

    if(!file)
    {
        LOG_ERROR((dipo, "Unable to open Icon present under %s", iconPath));
        err = kPathErr;
    }
    else
    {
        if(!file.eof())
        {
            // read file header
            char buffer[PNG_FILE_HEADER_BYTES];
            file.read(buffer, PNG_FILE_HEADER_BYTES);

            //check whether valid PNG file
            if(buffer[0] == (char)0x89 && buffer[1] == 'P' && buffer[2] == 'N'
                    && buffer[3] == 'G')
            {
                //read width and height
                file.read((char *)width, PNG_FILE_WIDTH_BYTES);
                file.read((char *)height, PNG_FILE_HEIGHT_BYTES);

                *width  = ntohl(*width);
                *height = ntohl(*height);

                file.close();
            }
            else
            {
                LOG_ERROR((dipo, "OemIcon present under %s is not of PNG type!!", iconPath));
                err = kFormatErr;
                file.close();
            }
        }
        else
        {
            LOG_ERROR((dipo, "OemIcon present under %s is not valid file", iconPath));
            err = kUnsupportedDataErr;
        }
    }

    return err;
}

} } /* namespace adit { namespace carplay */
